<?php

namespace App\Extensions\MarketingBot\System\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class ShopifyOrder extends Model
{
    protected $table = 'ext_shopify_orders';

    protected $fillable = [
        'shopify_store_id',
        'shopify_order_id',
        'order_number',
        'customer_email',
        'customer_name',
        'customer_phone',
        'total_price',
        'subtotal_price',
        'total_tax',
        'total_discounts',
        'currency',
        'financial_status',
        'fulfillment_status',
        'total_items',
        'line_items',
        'shipping_address',
        'billing_address',
        'utm_source',
        'utm_medium',
        'utm_campaign',
        'landing_site',
        'referring_site',
        'shopify_created_at',
        'shopify_updated_at',
    ];

    protected $casts = [
        'total_price' => 'decimal:2',
        'subtotal_price' => 'decimal:2',
        'total_tax' => 'decimal:2',
        'total_discounts' => 'decimal:2',
        'total_items' => 'integer',
        'line_items' => 'array',
        'shipping_address' => 'array',
        'billing_address' => 'array',
        'shopify_created_at' => 'datetime',
        'shopify_updated_at' => 'datetime',
    ];

    /**
     * Get the store that owns this order
     */
    public function store(): BelongsTo
    {
        return $this->belongsTo(ShopifyStore::class, 'shopify_store_id');
    }

    /**
     * Get campaign conversions for this order
     */
    public function conversions(): HasMany
    {
        return $this->hasMany(CampaignConversion::class);
    }

    /**
     * Check if order is paid
     */
    public function isPaid(): bool
    {
        return in_array($this->financial_status, ['paid', 'partially_paid']);
    }

    /**
     * Check if order is fulfilled
     */
    public function isFulfilled(): bool
    {
        return $this->fulfillment_status === 'fulfilled';
    }

    /**
     * Check if order came from a campaign
     */
    public function hasAttribution(): bool
    {
        return !empty($this->utm_campaign) || !empty($this->utm_source);
    }

    /**
     * Get the attributed campaign
     */
    public function getAttributedCampaign(): ?MarketingCampaign
    {
        if (!$this->utm_campaign) {
            return null;
        }

        // Try to find campaign by UTM parameter
        return MarketingCampaign::where('id', $this->utm_campaign)
            ->orWhere('name', 'like', '%' . $this->utm_campaign . '%')
            ->first();
    }

    /**
     * Scope for orders from a specific date range
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('shopify_created_at', [$startDate, $endDate]);
    }

    /**
     * Scope for orders with campaign attribution
     */
    public function scopeWithAttribution($query)
    {
        return $query->whereNotNull('utm_campaign');
    }
}
